<?php
if (! class_exists ( 'WC_Braintree_Payment_Gateway' )) {
	return;
}
class WC_PayPal_Payment_Gateway extends WC_Braintree_Payment_Gateway {
	const ID = 'braintree_paypal_payments';
	
	/**
	 * Nonce id contained in the $_POST.
	 *
	 * @var string
	 */
	public static $nonce_id = 'braintree_paypal_payments_nonce';
	
	/**
	 * Device data field key located in $_POST.
	 *
	 * @var string
	 */
	public static $device_data_id = 'braintree_paypal_device_data';
	
	/**
	 *
	 * @var boolean
	 */
	private $cart_checkout_enabled;
	
	/**
	 *
	 * @var boolean
	 */
	private $checkout_banner_enabled;
	
	/**
	 * The key used in $_POST to identify the selected payment method token.
	 *
	 * @var string
	 */
	public static $token_id = 'braintree_paypal_payments_method_token';
	public static $save_method_name = 'bfwc_save_paypal_method';

	public function __construct() {
		$this->enabled = $this->settings[ 'enabled' ] = (bwc_is_paypal_enabled () ? 'yes' : 'no');
		$this->title = bt_manager ()->get_option ( 'paypal_gateway_title' );
		$this->id = static::ID;
		$this->method_title = __ ( 'PayPal Braintree Payment Gateway', 'braintree-payments' );
		$this->has_fields = true;
		$this->actions ();
		$this->set_supports ();
		$this->init_settings ();
		// $this->icon = 'https://www.paypalobjects.com/webstatic/en_US/i/buttons/PP_logo_h_100x26.png';
		$this->icon = 'https://www.paypalobjects.com/webstatic/i/logo/rebrand/ppcom.svg';
		$this->cart_checkout_enabled = bwc_paypal_cart_checkout_enabled ();
		$this->checkout_banner_enabled = bwc_paypal_checkout_banner_enabled ();
	}

	public static function init() {
		add_filter ( 'bwc_add_payment_gateways', __CLASS__ . '::add_braintree_gateway' );
		
		add_filter ( 'woocommerce_payment_gateways', __CLASS__ . '::add_gateway' );
		
		add_filter ( 'woocommerce_saved_payment_methods_list', __CLASS__ . '::saved_payment_method_list', 10, 2 );
		
		add_action ( 'wp_loaded', __CLASS__ . '::maybe_delete_payment_method' );
		
		/**
		 *
		 * @since 2.6.26
		 */
		add_action ( 'wp_loaded', __CLASS__ . '::make_default_method' );
		
		add_filter ( 'woocommerce_payment_complete_order_status', __CLASS__ . '::maybe_update_order_status', 99, 3 );
		
		add_action ( 'woocommerce_subscription_failing_payment_method_updated_' . static::ID, __CLASS__ . '::update_failing_payment_method', 10, 2 );
		
		add_action ( 'bfwc_before_process_order_' . static::ID, __CLASS__ . '::maybe_save_payment_method' );
	}

	public static function add_braintree_gateway($gateways) {
		$gateways[] = __CLASS__;
		return $gateways;
	}

	public static function add_gateway($gateways) {
		$gateways[] = __CLASS__;
		return $gateways;
	}

	public function payment_fields() {
		if (bwc_is_checkout () || bwcs_is_change_payment_method () || bfwcs_is_change_payment_method () || bfwcs_is_pay_for_subscription_request ()) {
			
			bwc_get_template ( 'checkout/paypal-checkout.php', array( 
					'gateway' => $this 
			) );
		} elseif (is_add_payment_method_page ()) {
			
			bwc_get_template ( 'paypal.php', array( 
					'has_methods' => false,
					'gateway' => $this 
			) );
		}
	}

	public static function saved_payment_method_list($saved_methods, $user_id) {
		if (bwc_is_paypal_enabled ()) {
			$methods = bwc_get_user_paypal_payment_methods ( $user_id );
			$saved_methods = bwc_saved_payment_methods_list ( $saved_methods, $methods, static::ID );
		}
		return $saved_methods;
	}

	public function set_supports() {
		parent::set_supports ();
		if (($key = array_search ( 'bfwc_credit_card_form', $this->supports )) !== false) {
			unset ( $this->supports[ $key ] );
		}
		$this->supports[] = 'bfwc_cart_checkout';
		$this->supports[] = 'bfwc_autofill_checkout_fields';
	}

	/**
	 * Always add device data for PayPal transactions.
	 *
	 * @param unknown $attribs        	
	 */
	public static function add_device_data(&$attribs) {
		$attribs[ 'deviceData' ] = stripslashes ( self::get_request_param ( static::$device_data_id ) );
		
		return $attribs;
	}

	public static function order_button_text($text) {
		$payment_method = $_POST[ 'payment_method' ];
		if ($payment_method === static::ID) {
			$text = __ ( 'Checkout With PayPal', 'braintree-payments' );
		}
		return $text;
	}

	public function validate_fields() {
		if (strlen ( self::get_request_param ( static::$nonce_id ) ) == 0 && strlen ( self::get_request_param ( static::$token_id ) ) == 0) {
			if (is_checkout ()) {
				wc_add_notice ( __ ( 'Before payment can be made, you must go through the PayPal checkout process or select a saved payment method.', 'braintree-payments' ), 'error' );
			} elseif (is_add_payment_method_page ()) {
				wc_add_notice ( __ ( 'Please choose a PayPal payment method to save.', 'braintree-payments' ), 'error' );
			}
		}
	}

	public function get_key_mappings() {
		return array( 
				'enabled' => 'enable_paypal' 
		);
	}

	/**
	 *
	 * @param array $data        	
	 * @param WP_Error $errors        	
	 */
	public static function check_for_validation_errors($data, $errors) {
		if (isset ( $_POST[ 'bfwc-paypal-cart-checkout' ] ) && wp_verify_nonce ( $_POST[ 'bfwc-paypal-cart-checkout' ], 'bfwc-paypal-cart-checkout' )) {
			if (sizeof ( $errors->get_error_messages () ) > 0) {
				wc_maybe_define_constant ( 'WOOCOMMERCE_CHECKOUT', false );
			}
		}
	}

	public function order_success($order) {
		unset ( WC ()->session->paypal_response );
		return parent::order_success ( $order );
	}

	public function cart_fields() {
		bwc_get_template ( 'cart/paypal-checkout.php' );
		braintree_nonce_field ( static::$nonce_id );
		braintree_device_data_field ( static::$device_data_id );
	}

	/**
	 * Returns true if the gateway has cart checkout enabled.
	 *
	 * @return boolean
	 */
	public function is_cart_checkout_enabled() {
		return $this->cart_checkout_enabled;
	}

	public function checkout_banner_enabled() {
		return $this->checkout_banner_enabled;
	}
}
WC_PayPal_Payment_Gateway::init ();