<?php
class Braintree_Gateway_Kount_Controller
{

	public static function init()
	{
		$class = new self();
		
		$class->actions();
	}

	public function actions()
	{
		add_action( 'rest_api_init', array (
				$this, 
				'register_route' 
		) );
		
		add_action( 'bfwc_api_kount_event_WORKFLOW_STATUS_EDIT', array (
				$this, 
				'status_update' 
		), 10, 2 );
		add_action( 'bfwc_api_kount_event_WORKFLOW_NOTES_ADD', array (
				$this, 
				'notes_added' 
		), 10, 2 );
	}

	public function register_route()
	{
		register_rest_route( 'braintree-gateway/v1/', 'kount\/(?P<kount_key>[\w_-]+)', array (
				'methods' => WP_REST_Server::EDITABLE, 
				'callback' => array (
						$this, 
						'process_event' 
				) 
		) );
		
		register_rest_route( 'bfwc/settings/kount/', 'api_key/', array (
				'methods' => WP_REST_Server::CREATABLE, 
				'callback' => array (
						$this, 
						'create_api_key' 
				), 
				'permission_callback' => function ()
				{
					if ( ! current_user_can( 'administrator' ) ) {
						return new WP_Error( 'auth_error', __( 'Invalid permissions. Only admins allowed.', 'braintree-payments' ) );
					}
					return true;
				} 
		) );
	}

	/**
	 *
	 * @param WP_REST_Request $request        	
	 */
	public function create_api_key( $request )
	{
		$response = new WP_REST_Response();
		$new_key = $this->generate_key();
		update_option( 'bfwc_kount_key', $new_key );
		
		$response->set_status( 200 );
		$response->set_data( array (
				'url' => get_rest_url( null, 'braintree-gateway/v1/kount/' . $new_key ) 
		) );
		return $response;
	
	}

	public function generate_key()
	{
		return 'api_' . wc_rand_hash();
	}

	/**
	 *
	 * @param WP_REST_Request $request        	
	 */
	public function process_event( $request )
	{
		$response = new WP_REST_Response();
		
		try {
			
			/*
			 * commented out in favor of using API keys instead of IP addresses.
			 * $ip_address = $_SERVER [ 'REMOTE_ADDR' ];
			 *
			 * $accepted_ips = array (
			 * '209\.81\.12\.[\d]+',
			 * '64\.128\.91\.[\d]+',
			 * '64\.128\.87\.[\d]+'
			 * );
			 *
			 * $accepted_ips = implode( '|', $accepted_ips );
			 *
			 * if ( ! preg_match( '#' . $accepted_ips . '#', $ip_address ) ) {
			 * throw new Exception( sprintf( __( 'IP address %s is not allowed.', 'braintree-payments' ), $ip_address ), 403 );
			 * }
			 */
			
			$kount_key = $request->get_param( 'kount_key' );
			
			$saved_key = get_option( 'bfwc_kount_key', false );
			
			if ( $kount_key == null || ! hash_equals( $saved_key, $kount_key ) ) {
				throw new Exception( __( 'Invalid API key provided.', 'braintree-payments' ), 403 );
			}
			
			$xml = $request->get_body();
			
			bt_manager()->info( 'kount xml: ' . base64_encode( $xml ) );
			
			$document = new DOMDocument();
			if ( @$document->loadXML( $xml ) ) {
				
				$element = $document->documentElement;
				
				$event_elements = $element->getElementsByTagName( 'event' );
				
				$total_events = intval( $element->getAttribute( 'total' ) );
				
				foreach ( $event_elements as $event ) {
					
					/**
					 *
					 * @var $event DOMElement
					 */
					$event;
					
					$name = $event->getElementsByTagName( 'name' )->item( 0 )->nodeValue;
					
					$order_id = $event->getElementsByTagName( 'key' )->item( 0 )->getAttribute( 'order_number' );
					
					$order = wc_get_order( $order_id );
					
					if ( $order ) {
						$attribs = array (
								'name' => $name, 
								'key' => $event->getElementsByTagName( 'key' )->item( 0 )->nodeValue, 
								'order_id' => $event->getElementsByTagName( 'key' )->item( 0 )->getAttribute( 'order_number' ), 
								'old_value' => $event->getElementsByTagName( 'old_value' )->item( 0 )->nodeValue, 
								'new_value' => $event->getElementsByTagName( 'new_value' )->item( 0 )->nodeValue, 
								'agent' => $event->getElementsByTagName( 'agent' )->item( 0 )->nodeValue, 
								'occurred' => $event->getElementsByTagName( 'occurred' )->item( 0 )->nodeValue 
						);
						
						do_action( 'bfwc_api_kount_event_' . $name, $order, $attribs, $request );
					}
				}
			
			} else {
				throw new Exception( __( 'Invalid xml format.', 'braintree-payments' ), 400 );
			}
			$response->set_status( 200 );
			return $response;
		} catch( Exception $e ) {
			$response->set_status( $e->getCode() );
			$message = sprintf( __( 'Exception encountered. Reason: %s', 'braintree-payments' ), $e->getMessage() );
			$response->set_data( array (
					'message' => $message 
			) );
			bt_manager()->error( $message );
			return $response;
		}
	}

	/**
	 * Update the status of the order.
	 *
	 * @param WC_Order $order        	
	 * @param array $attribs        	
	 */
	public function status_update( $order, $attribs )
	{
		$status = $attribs [ 'new_value' ];
		$order_id = $attribs [ 'order_id' ];
		
		$current_status = $order->get_status();
		$new_status = $current_status;
		// A,R,E,D
		switch( $status ) {
			case 'A' :
				// $new_status = $order->needs_processing() ? 'processing' : 'completed';
				$new_status = null;
				
				add_filter( 'woocommerce_valid_order_statuses_for_payment_complete', array (
						$this, 
						'add_valid_order_statuses' 
				) );
				
				$order->payment_complete();
				
				remove_filter( 'woocommerce_valid_order_statuses_for_payment_complete', array (
						$this, 
						'add_valid_order_statuses' 
				) );
				
				break;
			case 'E' :
				$new_status = 'kount-escalate';
				break;
			case 'D' :
				$new_status = 'cancelled';
				
				$transaction = $this->fetch_transaction( $order->get_transaction_id() );
				
				$action = bt_manager()->get_option( 'kount_decline_action' );
				
				if ( $transaction != false ) {
					if ( $action === 'cancel_transaction' ) {
						switch( $transaction->status ) {
							case Braintree_Transaction::AUTHORIZED :
							case Braintree_Transaction::SUBMITTED_FOR_SETTLEMENT :
								$this->void_transaction( $order );
								break;
							case Braintree_Transaction::SETTLED :
								$this->refund_transaction( $order );
								break;
						}
					}
				} else {
					$order->add_order_note( sprintf( __( 'Transaction %s was not found in the Braintree %s system.', 'braintree-payments' ), $order->get_transaction_id(), bt_manager()->get_environment() ) );
				}
				break;
			case 'R' :
				$new_status = 'kount-review';
				break;
		}
		
		if ( ! is_null( $new_status ) && $new_status !== $current_status && apply_filters( 'bfwc_kount_api_can_change_status', true, $order, $transaction ) ) {
			$order->update_status( $new_status );
		}
	}

	/**
	 *
	 * @param WC_Order $order        	
	 * @param array $attribs        	
	 */
	public function notes_added( $order, $attribs )
	{
		if ( ! empty( $attribs [ 'new_value' ] ) ) {
			$order->add_order_note( sprintf( __( 'Kount note: %s', 'braintree-payments' ), $attribs [ 'new_value' ] ) );
		}
	}

	/**
	 * Void the transaction in Braintree.
	 *
	 * @param WC_Order $order        	
	 */
	public function void_transaction( $order )
	{
		try {
			$result = Braintree_Transaction::void( $order->get_transaction_id() );
			
			if ( $result->success ) {
				$order->add_order_note( __( 'Transaction voided in Braintree. Reason: Kount transaction status updated to Decline.', 'braintree-payments' ) );
			} else {
				$order->add_order_note( sprintf( __( 'Transaction not voided in Braintree. Reason: %s', 'braintree-payments' ), $result->message ) );
			}
		} catch( \Braintree\Exception $e ) {
			
			$order->add_order_note( sprintf( __( 'Transaction not voied in Braintree. Reason: %s', 'braintree-payments' ), $e->getMessage() ) );
		}
	}

	/**
	 * Refund the order in Braintree.
	 *
	 * @param WC_Order $order        	
	 */
	public function refund_transaction( $order )
	{
		$result = wc_create_refund( array (
				'amount' => $order->get_total(), 
				'reason' => __( 'Order refunded.', '' ), 
				'order_id' => bwc_get_order_property( 'id', $order ), 
				'line_items' => array (), 
				'refund_payment' => true, 
				'restock_items' => true 
		) );
		
		if ( is_wp_error( $result ) ) {
			$order->add_order_note( sprintf( __( 'Refund via Kount API not possible. Reason: %s.', 'braintree-payments' ), $result->get_error_message() ) );
		}
	}

	/**
	 *
	 * @param string $id        	
	 * @return \Braintree\Transaction|bool
	 */
	public function fetch_transaction( $id )
	{
		try {
			$transaction = Braintree_Transaction::find( $id );
			return $transaction;
		} catch( \Braintree\Exception $e ) {
			return false;
		}
	}

	public function add_valid_order_statuses( $statuses )
	{
		$statuses [] = 'kount-review';
		$statuses [] = 'kount-escalate';
		return $statuses;
	}
}
Braintree_Gateway_Kount_Controller::init();